import Database from "better-sqlite3";
import { Author, Quote } from "./types";

const db = Database("../quotes.db", { fileMustExist: true });

export function getRandomQuote() {
  const queryResult = db
    .prepare(
      `
      SELECT q.id as id, q.text as text, a.name as authorName, a.id as authorId
      FROM
        quote as q
        JOIN author as a ON q.authorId = a.id
      ORDER BY RANDOM()
      LIMIT 1
    `
    )
    .get();
  const randomQuote: Quote = {
    id: queryResult.id,
    text: queryResult.text,
    author: {
      id: queryResult.authorId,
      name: queryResult.authorName,
    },
  };

  return randomQuote;
}

export function getQuotesOfAuthor(author: Author) {
  const queryResult = db
    .prepare(
      `
        SELECT *
        FROM quote
        WHERE authorId = ?
      `
    )
    .all(author.id);

  const quotes = queryResult.map(
    (x): Quote => ({ id: x.id, text: x.text, author })
  );
  return quotes;
}

export function addQuote(quote: string, authorName: string): Quote {
  let existingAuthor = getAuthorByName(authorName);
  if (!existingAuthor) existingAuthor = addAuthor(authorName);

  const result = db
    .prepare(`INSERT INTO quote (text, authorId) VALUES(?, ?)`)
    .run(quote, existingAuthor.id);
  const added = getQuoteById(Number(result.lastInsertRowid))!;
  return added;
}

export function addAuthor(name: string): Author {
  const result = db.prepare(`INSERT INTO author (name) VALUES(?)`).run(name);
  const added = getAuthorById(Number(result.lastInsertRowid))!;
  return added;
}

export function getAuthorById(id: number): Author | null {
  const result = db.prepare(`SELECT * FROM author WHERE id = ?`).get(id);
  if (!result) return null;

  return {
    id: result.id,
    name: result.name,
  };
}

export function getQuoteById(id: number): Quote | null {
  const result = db
    .prepare(
      `
      SELECT q.id as id, q.text as text, a.name as authorName, a.id as authorId
      FROM
        quote as q
        JOIN author as a ON q.authorId = a.id
      WHERE q.id = ?`
    )
    .get(id);
  if (!result) return null;

  return {
    id: result.id,
    text: result.text,
    author: {
      id: result.authorId,
      name: result.authorName,
    },
  };
}

export function getAuthorByName(name: string): Author | null {
  const result = db.prepare(`SELECT * FROM author WHERE name = ?`).get(name);
  if (!result) return null;

  return {
    id: result.id,
    name: result.name,
  };
}
