import { HttpClient } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { catchError, map, Observable, of } from 'rxjs';
import { Author, Quote } from './api.types';

@Injectable({
  providedIn: 'root',
})
export class QuotesService {
  backendUrl = 'http://localhost:3001';

  constructor(private client: HttpClient) {}

  getRandomQuote() {
    return this.client.get<Quote>(`${this.backendUrl}/quotes/random`, {
      withCredentials: true,
    });
  }

  getQuotesOfAuthor(author: Author): Observable<Quote[]> {
    return this.client.get<Quote[]>(
      `${this.backendUrl}/authors/${author.id}/quotes`,
      { withCredentials: true }
    );
  }

  createQuote(text: string, authorName: string): Observable<Quote> {
    return this.client.post<Quote>(
      `${this.backendUrl}/quotes`,
      {
        quote: text,
        authorName,
      },
      { withCredentials: true }
    );
  }

  checkLogin(): Observable<boolean> {
    return this.client.get<boolean>(`${this.backendUrl}/check-login`, {
      withCredentials: true,
    });
  }

  login(credentials: {
    username: string;
    password: string;
  }): Observable<boolean> {
    return this.client
      .post(`${this.backendUrl}/login`, credentials, {
        withCredentials: true,
      })
      .pipe(
        map(() => true),
        catchError(() => of(false))
      );
  }

  logout() {
    return this.client
      .post(`${this.backendUrl}/logout`, null, {
        withCredentials: true,
      })
      .subscribe(() => location.reload());
  }
}
